/* 
 *  ResourceDataGUI.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import java.util.List;

import com.avaya.ccs.api.enums.ContactType;

import javafx.collections.FXCollections;
import javafx.collections.ObservableList;
import javafx.event.ActionEvent;
import javafx.geometry.Pos;
import javafx.scene.control.Button;
import javafx.scene.control.MenuItem;
import javafx.scene.control.TextArea;
import javafx.scene.layout.HBox;
import javafx.scene.text.Text;

/**
 * This class is responsible for updating all the GUI components related to
 * resources.
 */
public class ResourceDataGUI extends ComboDataGUI {

    private final HBox hbox;
    private final Button originateButton;

    private final Button callSupervisorButton;
    private TextArea numberToDial;
    private final Text resourceState;
    private static final Logger LOG = Logger.getLogger(ResourceDataGUI.class);

    public ResourceDataGUI(ResourceExecutor resExe, String css) {
        super(LOG);
        setDetailsForm(new ResourceDetailsForm(resExe, css, this));

        hbox = new HBox();
        hbox.getStyleClass().add("hbox");

        originateButton = new Button("Originate");
        originateButton.getStyleClass().add("agentButton");
        originateButton.setOnAction((ActionEvent e) -> {
            String methodName = "OriginateButton() ";
            String number = numberToDial.getText();
            ResourceData resource = getCurrentResourceSelection();
            if (number != null && !number.isEmpty()) {
                LOG.info(methodName + "Number:" + number + " ResourceID:" + resource.getId());
                resExe.originateCall(number, resource.getId());
            } else {
                LOG.info(methodName + "number to dial is empty or null, ignore button press");
            }
        });
   
        callSupervisorButton = new Button("Call Supervisor");
        callSupervisorButton.getStyleClass().add("agentButton");
        callSupervisorButton.setOnAction((ActionEvent e) -> {
            String methodName = "CallSupervisorButton() ";
            ResourceData resource = getCurrentResourceSelection();
            LOG.info(methodName + " ResouceID:" + resource.getId());
            resExe.callSupervisor(resource.getId());
        });

        numberToDial = new TextArea();
        numberToDial.getStyleClass().add("numberToDial");

        resourceState = new Text();
        hbox.getChildren().addAll(getItemSelection(), originateButton, numberToDial, callSupervisorButton, 
                resourceState);
        hbox.setAlignment(Pos.CENTER_LEFT);

        redrawControls();
    }

    public MenuItem getMenuitem() {
        return getDetailsForm().getMenuItem();
    }

    @Override
    protected void redrawControls() {
        String methodName = "redrawControls() ";
        ResourceData resource = getCurrentResourceSelection();
        LOG.trace(methodName + "ID:" + resource.getId());
        originateButton.setDisable(!resource.canOriginate());
        callSupervisorButton.setDisable(!resource.canCallSupervisor());
        numberToDial.setDisable(!resource.canOriginate());
        resourceState.setText(resource.getState().name());
    }

    public void UpdateResourceData(List<ResourceData> data) {
        // Need to convert to use data objects before setting....
        ObservableList<Data> newData = FXCollections.observableArrayList();
        newData.addAll(data);
        this.update(newData);
    }

    public void UpdateResourceData(ResourceData data) {
        this.update(data);
    }

    public HBox getAddressButtons() {
        return hbox;
    }

    // Return the current user selection or a dummy value if there is none....
    private ResourceData getCurrentResourceSelection() {
        String methodName = "getCurrentResourceSelection() ";
        Data currentResource = getCurrentItemSelection();

        if (currentResource != null) {
            return (ResourceData) currentResource;
        } else {
            LOG.trace(methodName + "current selection is NULL, returning blank");
            return new ResourceData(null);
        }
    }
}
